/* $Id: srcrealtime.c,v 1.20 1998/07/22 23:51:10 donwm Exp $ */
/* Copyright (C) 1997 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Don Mathiesen */

/* This file contains code which tests the E1434 or E1432/33 SRC Option
   for realtime performance under various modes of operation. */

#include <stdio.h>		/* For printf */
#include <unistd.h>		/* For sleep */
#include <sicl.h>		/* For SICL functions */
#include <math.h>		
#include "e1432.h"
#include "err1432.h"
#include <sys/termio.h>		/* For key press routine */

#define TIMEOUT		10.0f
#define	SRC_MAX		5
/*#define DPRINT */

/* Wrap this around all the many function calls which might fail */
#define	DEBUG(s)	s
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s != 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s != 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (0)
#endif
#define CHECK_RT(func)	\
do {\
    int _s = (func);\
    if (_s !=0)\
    {\
        if (_s == ERR1432_SRC_REALTIME_RESTRICTION)\
        {\
           if (underrun == 0)\
           {\
	      DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
           }\
           underrun = 1;\
        }\
        else\
        {\
	   DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
           return _s;\
        }\
    }\
} while (0)


#define GET_INT(parm, msg)                      \
{                                               \
    rtn = sscanf(optarg, "%f", &ftmp);          \
    if ( rtn == 1 )                             \
    {                                           \
        (parm) = (LONGSIZ32)(ftmp + .5);        \
    }                                           \
    else                                        \
    {                                           \
        (void) fprintf(stderr, msg, optarg);    \
        exit(2);                                \
    }                                           \
}

#define GET_FLOAT(parm, msg)                    \
{                                               \
    rtn = sscanf(optarg, "%f", &(parm));        \
    if ( rtn != 1 )                             \
    {                                           \
        (void) fprintf(stderr, msg, optarg);    \
        exit(2);                                \
    }                                           \
}

/******************* arb source routines **************************/
static int verbose = 0;
static int rt_check = 0;

static struct timeval tv1, tv2;
static struct timezone tz;
static double time1;
static FLOATSIZ32 testtime_max;

#define SIG_LENGTH_MAX 150000
#define ARBXFRSIZEMAX 4096

static LONGSIZ32 arb_siglength;
static SHORTSIZ16 sigtype;
static LONGSIZ32 arb_sigperiod;

static LONGSIZ32 src_sigbuff1_a[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff2_a[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff3_a[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff4_a[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff5_a[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff1_b[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff2_b[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff3_b[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff4_b[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff5_b[SIG_LENGTH_MAX];

static LONGSIZ32 *src_sigbuffptrs_a[5];
static LONGSIZ32 *src_sigbuffptrs_b[5];

static int rd4data_set_count, rd4data_clr_count;
static SHORTSIZ16 arb_xfrmode, arb_xfrmodewait;

static SHORTSIZ16 source_mode = E1432_SOURCE_MODE_SINE, arb_mode;
static SHORTSIZ16 source_buffer_mode = E1432_SRCBUFFER_CONTINUOUS;
static LONGSIZ32 arb_srcbufsize;
static LONGSIZ32 arb_xfrsize = ARBXFRSIZEMAX;
static LONGSIZ32 arb_xfrsize2, arb_srcbufleft;

void
src_sig(LONGSIZ32 siglength, SHORTSIZ16 sigperiod, long *sigbuff)
{

    int i;

    for(i=0; i < siglength; i++)
    {
       sigbuff[i] = (long)((float) 0x7fffffffL *
			sin(2.0*M_PI*(float)i/((float)sigperiod)));
    }
}

void
xfr_src_sig(E1432ID lhw, SHORTSIZ16 chanID, long *sigbuff, 
		LONGSIZ32 xfrlength, SHORTSIZ16 mode)
{
    
    LONGSIZ32 i, maxtries;
    SHORTSIZ16 error;
    SHORTSIZ16 status;
    LONGSIZ32 xfrbuf_wds, srcbuf_state_AB, srcbuf_state_A, srcbuf_state_B;

    /* write xfrlength words to the substrate */
    i = 0;
    maxtries = 100;
    for (i=0; i < maxtries; i++)
    {
        if(verbose) printf("xfr_src_sig: try %d, Ch 0x%x  xfrlength %d, mode %d\n", i,
	       chanID, xfrlength, mode);
       status = e1432_check_src_arbrdy(lhw, chanID, mode);
       if (status == 1)
       {
	  error = e1432_write_srcbuffer_data(lhw, chanID, sigbuff,
			(SHORTSIZ16) xfrlength, mode);
	  if(error)
	  {
	     printf("Error %d in e1432_write_srcbuffer_data\n", error);
	     e1432_get_src_arbstates(lhw, chanID, &xfrbuf_wds,
			&srcbuf_state_AB, &srcbuf_state_A, &srcbuf_state_B);
	     printf("xfrbuf_wds %d, srcbuf_state_AB 0x%x, \
		srcbuf_state_A 0x%x, srcbuf_state_B 0x%x\n",
		xfrbuf_wds, srcbuf_state_AB,srcbuf_state_A,srcbuf_state_B);
	     status = e1432_check_src_arbrdy(lhw, chanID, mode);
	     printf("xfr_src_sig: status 0x%x\n", status);
	  }
	  return;
     
       }
    }
    if (i== maxtries)
    {
        printf("xfr_src_sig: Ch 0x%x  timed out \n", chanID);
        e1432_get_src_arbstates(lhw, chanID, &xfrbuf_wds, \
              &srcbuf_state_AB, &srcbuf_state_A, &srcbuf_state_B);
        printf("xfrbuf_wds %d, srcbuf_state_AB 0x%x, srcbuf_state_A 0x%x, \
              srcbuf_state_Bxf 0x%x\n",xfrbuf_wds,srcbuf_state_AB,
              srcbuf_state_A, srcbuf_state_B);

        status = e1432_check_src_arbrdy(lhw, chanID, mode);
        printf("xfr_src_sig: status 0x%x\n",status);
        return;
    }

}



/*****************************************************************************
 *
 * Check for key being pressed.  Return  if key pressed, 0 if no key.
 *
 ****************************************************************************/
static char key_pressed(void)
{
  char ch;
  LONGSIZ32 numChars;

  ioctl(0, FIONREAD, &numChars);

  if(numChars) {
    read(0, &ch, 1);
    return(ch);
  }

  return(0);
}


/* Make these global so the interrupt handler can get to them easily */
static E1432ID hw;
static SHORTSIZ16 chan_list[E1432_INPUT_CHANS];
static SHORTSIZ16 group;
static int chan_count;
static int underrun=0;

static SHORTSIZ16 src_list[SRC_MAX];
static SHORTSIZ16 src_group;
static int src_count;

static INST irq_id;
static int irq_laddr;
static int irq_flag;

static int
srcstatus(void)
{
    int     i, temp;

    /* Put this after checking for src_arbrdy, so as to service arb data
       first and as quickly as possible.

      Only one interrupt per measurement for each of these events.
      The check src status conditions are latched until the source is
      turned off, then it is cleared.
      The source status irq is reenabled when the source is turned on. 
     */

    /* check source status */
    for (i = 0; i < src_count; i++)
    {
        if (e1432_check_src_shutdown(hw, src_list[i]) == 1)
            (void) printf("    src_shutdown from  0x%x\n", src_list[i]);
        if (e1432_check_src_overload(hw, src_list[i]) == 1)
            (void) printf("    src_overload from 0x%x\n", src_list[i]);
	/* Don't check for overreads when doing realtime check,
	   because the check for overreads can occasionally cause an
	   underrun to get missed. */
	if (!rt_check)
	    if (e1432_check_src_overread(hw, src_list[i]) == 1)
		(void) printf("    src_overread from 0x%x\n", src_list[i]);
        if(!underrun || !rt_check )
           if (underrun = e1432_check_src_underrun(hw, src_list[i]) == 1)
           {
	      if(!rt_check || verbose) printf("    src_underrun from 0x%x\n", 
						src_list[i]);
              CHECK(e1432_set_interrupt_mask(hw, src_group, 0 ));
	   }
    }

    if (source_mode == E1432_SOURCE_MODE_ARB)
       for (i = 0; i < src_count; i++)
       {
           if (e1432_check_src_arbrdy(hw, src_list[i], arb_xfrmode) == 1)
           {
		if(verbose) printf("   src_arbrdy 1 from %d\n", src_list[i]);
#ifdef DPRINT
               (void) printf("    src_arbrdy from %d\n", src_list[i]);
#endif
               /* down load signal */
               xfr_src_sig(hw, src_list[i], src_sigbuffptrs_a[i], arb_xfrsize, 
			arb_xfrmode);
           }
	   else
		if(verbose) printf("   src_arbrdy 0 from %d\n", src_list[i]);
       }

    return 0;
}

static int
irqstatus(void)
{
    LONGSIZ32 status;
    LONGSIZ32 xfrbuf_wds, srcbuf_state_AB, srcbuf_state_A, srcbuf_state_B;
    int i;

    /* check irq status */

    if (src_count != 0)
	 {
	     CHECK( e1432_read32_register(hw, src_list[0], 
			E1432_IRQ_STATUS2_REG, &status) );
	     if(verbose)
	        (void) printf("    E1432_IRQ_STATUS2_REG = %x\n", status);
	     CHECK( e1432_read32_register(hw, src_list[0], 
			E1432_SRC_STATUS_REG, &status) );
	     if(verbose)
	        (void) printf("    E1432_SRC_STATUS_REG = %x\n\n", status);

	     return 0;

 	 }
    else 
	{
	    if (chan_count != 0)
	    {
		CHECK( e1432_read32_register(hw, chan_list[0], 
				E1432_IRQ_STATUS2_REG, &status) );
		(void) printf("    E1432_IRQ_STATUS2_REG = %x\n\n", status);
	    }
	}

    printf("irqstatus: \n"); 
    if(source_mode == E1432_SOURCE_MODE_ARB)
    {
        printf("irqstatus: checking src_arbstates\n"); 
	for(i=0; i < src_count; i++)
	{
           (void) e1432_get_src_arbstates(hw, src_list[i], &xfrbuf_wds,
                                   &srcbuf_state_AB, &srcbuf_state_A,
                                   &srcbuf_state_B);
           printf("xfrbuf_wds = %d, srcbuf_state_AB = 0x%x, "
                  "srcbuf_state_A = 0x%x, srcbuf_state_B = 0x%x\n",
                  xfrbuf_wds, srcbuf_state_AB,
                  srcbuf_state_A, srcbuf_state_B);
	}
    }
    
    return 0;
}



static void
irq_handler(INST id, long reason, long sec)
{
    SHORTSIZ16 error;
    int i;

    if(verbose)
       DEBUG((void) printf("irq_handler called\n"));

    if (id != irq_id)
	(void) printf("Error: irq_handler got wrong id\n");
    if (reason != I_INTR_VXI_SIGNAL)
	(void) printf("Error: irq_handler got wrong reason\n");
    if ((sec & E1432_IRQ_STATUS_LADDR_MASK) != irq_laddr)
	(void) printf("Error: irq_handler got wrong laddr\n");

    irq_flag = 1;
    (void) irqstatus();
    if ((sec & E1432_IRQ_SRC_STATUS) != 0)    (void) srcstatus();
    if ((sec & (E1432_IRQ_SRC_STATUS)) == 0)
	(void) printf("Error: irq_handler got wrong sec\n");

    /* Do timeout and underrun checks inside handler, otherwise there
       may be a small window of time (due to SICL calls) when interrupts
       would always occurr (especially for arb) and it would never get
       to the code outside of the handler. 

       Should find a better way to do this as this slows down the 
       handler.
    */
    if(source_mode == E1432_SOURCE_MODE_ARB) 
    {
       (void) gettimeofday(&tv2, & tz);
       time1 = tv2.tv_sec - tv1.tv_sec + (tv2.tv_usec - tv1.tv_usec)/1000000.0;

       if(!rt_check || ((time1 <= testtime_max) && !underrun))
       {
          error = e1432_reenable_interrupt(hw, src_group);
	  if(error)
	     printf("Error %d from e1432_reenable_interrupt\n", error);
       }
    }
    else
    {
       error = e1432_reenable_interrupt(hw, src_group);
       if(error)
          printf("Error %d from e1432_reenable_interrupt\n", error);

    }
 
}

static int
irq_setup(int laddr, int *line)
{
    struct vxiinfo info;
    unsigned long slot0_laddr;
    char    addr[16];
    INST    id;
    int     i;

    /* Get the interrupt line to use */
    id = iopen("vxi");
    if (id == 0)
    {
	DEBUG((void) printf("Error: iopen returned %d\n", id));
	return -1;
    }
    CHECK(ivxibusstatus(id, I_VXI_BUS_LADDR, &slot0_laddr));
    CHECK(ivxirminfo(id, slot0_laddr, &info));
    for (i = 0; i < 8; i++)
	if (info.int_handler[i] != -1)
	{
	    *line = info.int_handler[i];
	    break;
	}
    if (i == 8)
    {
	DEBUG((void) printf("Error: no interrupt lines available\n"));
	return -1;
    }
    if(verbose)
       DEBUG((void) printf("Using VME interrupt line %d\n", *line));

    /* Set up the interrupt handler routine */
    irq_laddr = laddr;
    (void) sprintf(addr, "vxi,%d", irq_laddr);
    irq_id = iopen(addr);
    if (irq_id == 0)
    {
	DEBUG((void) printf("Error: iopen returned %d\n", irq_id));
	return -1;
    }
    CHECK(ionintr(irq_id, irq_handler));
    CHECK(isetintr(irq_id, I_INTR_VXI_SIGNAL, 1));

    return 0;
}

int
main(int argn, char **argv)
{
    int     i, k, line;
    struct e1432_hwconfig hwconfig;
    SHORTSIZ16 laddr = 8;
    float amp;
    int opt;			/* really char returned */
    int rtn, error;
    float ftmp;
    LONGSIZ32 tmp=0;
    FLOATSIZ32 clock_freq = 65536;	/* Default value */
    FLOATSIZ32 span = 0;		/* Default value */
    int src_ch = 0;
    LONGSIZ32 temp_long;

    /* Realtime Test variables */
    FLOATSIZ32 rt_clock_freqs[6] = {65536, 64000, 51200, 50000, 48000, 40960};
    int rt_clock_freqs_num = 6; 
    int rt_pass = 0, rt_dec_factor = 1, rt_otf = 0;
    int rt_ramp = 0;
    char mode_ascii[8][80]={"SINE","BSINE","RAND","BRAND","RANDZ","BRANDZ",\
			    "ARB","BARB"};
    int rt_mode;

    FLOATSIZ32 src_dutycycle, src_ramptime;
    LONGSIZ32 src_blocksize;
    
    /* Arb variables */
    src_sigbuffptrs_a[0] = &src_sigbuff1_a[0];
    src_sigbuffptrs_a[1] = &src_sigbuff2_a[0];
    src_sigbuffptrs_a[2] = &src_sigbuff3_a[0];
    src_sigbuffptrs_a[3] = &src_sigbuff4_a[0];
    src_sigbuffptrs_a[4] = &src_sigbuff5_a[0];
    src_sigbuffptrs_b[0] = &src_sigbuff1_b[0];
    src_sigbuffptrs_b[1] = &src_sigbuff2_b[0];
    src_sigbuffptrs_b[2] = &src_sigbuff3_b[0];
    src_sigbuffptrs_b[3] = &src_sigbuff4_b[0];
    src_sigbuffptrs_b[4] = &src_sigbuff5_b[0];

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    e1432_trace_level(0);

    /* Print out command line arguments used */
    if(verbose)
    {
       printf("Command line args used:\n");
       for(i=0; i<argn; i++)
          printf(" %s", argv[i]);
       printf("\n\n");
    }

    testtime_max = 10.0;

    while ( ( opt = getopt(argn, argv, "c:d:f:m:ors:vx:B:L:RT:") ) != EOF )
    {
	switch (opt)
        {
	   case 'c':
	      GET_INT(tmp, " number channels %s not converted");
	      src_ch = tmp;
	      break;
	   case 'd':
	      GET_INT(tmp, " decimation factor %s not converted");
	      rt_dec_factor = tmp;
	      break;
	   case 'f':
	      GET_FLOAT(clock_freq, " clock_freq %s not converted");
	      break;
	   case 'm':
	      GET_INT(tmp, " source_mode %s not converted");
              rt_mode = tmp;
	      switch(tmp)
	      {
		case 1:
		   source_mode = E1432_SOURCE_MODE_SINE;
		   break;
		case 2:
		   source_mode = E1432_SOURCE_MODE_BSINE;
		   break;
		case 3:
		   source_mode = E1432_SOURCE_MODE_RAND;
		   break;
		case 4:
		   source_mode = E1432_SOURCE_MODE_BRAND;
		   break;
		case 5:
		   source_mode = E1432_SOURCE_MODE_RANDZ;
		   break;
		case 6:
		   source_mode = E1432_SOURCE_MODE_BRANDZ;
		   break;
		case 7:
		   source_mode = E1432_SOURCE_MODE_ARB;
		   break;
		case 8:
		   source_mode = E1432_SOURCE_MODE_BARB;
		   break;
		default:
		   source_mode = E1432_SOURCE_MODE_SINE;
	      }
	      break;
	   case 'o':
	      rt_otf = 1;
	      break;
	   case 'r':
	      rt_check = 1;
	      break;
	   case 's':
	      GET_FLOAT(span, " span %s not converted");
	      break;
	   case 'v':
	      verbose = 1;
	      break;
	   case 'x':
	      GET_INT(tmp, " arb transfer size %s not converted");
	      arb_xfrsize = tmp;
	      break;
	   case 'B':
	      GET_INT(tmp, " source_buffer_mode %s not converted");
	      switch(tmp)
	      {
	      case 1:
		  source_buffer_mode = E1432_SRCBUFFER_CONTINUOUS;
		  arb_srcbufsize = 4096;
		  break;
	      case 2:
		  source_buffer_mode = E1432_SRCBUFFER_PERIODIC_A;
		  arb_srcbufsize = 4096;
		  break;
	      case 3:
		  source_buffer_mode = E1432_SRCBUFFER_PERIODIC_AB;
		  arb_srcbufsize = 4096;
		  break;
	      case 4:
		  source_buffer_mode = E1432_SRCBUFFER_CONTINUOUS;
		  arb_srcbufsize = 16384;
		  break;
	      case 5:
		  source_buffer_mode = E1432_SRCBUFFER_PERIODIC_A;
		  arb_srcbufsize = 16384;
		  break;
	      case 6:
		  source_buffer_mode = E1432_SRCBUFFER_PERIODIC_AB;
		  arb_srcbufsize = 16384;
		  break;
	      case 7:
		  source_buffer_mode = E1432_SRCBUFFER_CONTINUOUS;
		  arb_srcbufsize = 32768;
		  break;
	      case 8:
		  source_buffer_mode = E1432_SRCBUFFER_PERIODIC_A;
		  arb_srcbufsize = 32768;
		  break;
	      case 9:
		  source_buffer_mode = E1432_SRCBUFFER_PERIODIC_AB;
		  arb_srcbufsize = 32768;
		  break;
	      case 10:
		  source_buffer_mode = E1432_SRCBUFFER_CONTINUOUS;
		  arb_srcbufsize = 131072;
		  break;
	      case 11:
		  source_buffer_mode = E1432_SRCBUFFER_PERIODIC_A;
		  arb_srcbufsize = 131072;
		  break;
	      case 12:
		  source_buffer_mode = E1432_SRCBUFFER_PERIODIC_AB;
		  arb_srcbufsize = 131072;
		  break;
	      case 13:
		  source_buffer_mode = E1432_SRCBUFFER_ONESHOT;
		  arb_srcbufsize = 131072;
		  break;
	      default:
		  source_buffer_mode = E1432_SRCBUFFER_CONTINUOUS;
		  arb_srcbufsize = 131072;
	      }
	      break;
	   case 'L':
	      GET_INT(tmp, " number channels %s not converted");
	      laddr = tmp;
	      break;
	   case 'R':
	      rt_ramp = 1;
	      break;
	   case 'T':
	      GET_FLOAT(ftmp, " max time per test %s not converted");
	      testtime_max = ftmp;
	      break;
	   default:
	      printf("%s usage:\n", argv[0]);
	      printf("    -c Number of source chans to use\n");
	      printf("    -d decimation factor (realtime check)\n");
	      printf("    -f sample frequency\n");
	      printf("    -m source mode (1 SINE, 2 BSINE, 3 RAND, 4 BRAND,\n");
	      printf("       5 RANDZ, 6 BRANDZ, 7 ARB, 8 BARB) \n");
	      printf("    -o on the fly parameter changes (realtime check)\n");
	      printf("    -r perform realtime check\n");
	      printf("    -s span in Hz\n");
	      printf("    -v verbose\n");
	      printf("    -x arb transfer size\n");
	      printf("    -B src buffer mode (1 CONT_4K, 2 PER_A_4K, 3 PER_AB_4K,\n");
	      printf("       4 CONT_16K, 5 PER_A_16K, 6 PER_AB_16K,\n");
	      printf("       7 CONT_32K, 8 PER_A_32K, 9 PER_AB_32K,\n");
	      printf("       10 CONT_128K, 11 PER_A_128K, 12 PER_AB_128K,\n");
	      printf("       13 ARB_ONESHOT_128K)\n");
	      printf("    -L logical address\n");
	      printf("    -R ramp_rate <=3 sec. (realtime check)\n");
	      printf("    -T max time per test in seconds (float)\n");
	      exit(0);
        }
    }

    /* Use e1432_get_hwconfig to see if the module already has
       firmware.  If this errors, assume we need to install firmware,
       so use e1432_install to do it. */
    error = e1432_get_hwconfig(1, &laddr, &hwconfig);
    if (error != 0)
    {
	printf("downloading /opt/e1432/lib/sema.bin\n");
	error = e1432_install(1, &laddr, 0, "/opt/e1432/lib/sema.bin");
	if (error)
	{
	   printf("e1432_install failed, error: %d\n", error);
	   return -1;
	}
    }

    CHECK(e1432_print_errors(1));
    e1432_debug_level(0);
    e1432_trace_level(0);

    CHECK(e1432_assign_channel_numbers(1, &laddr, &hw));

    CHECK(e1432_get_hwconfig(1, &laddr, &hwconfig));

    /* No inputs used so no input group created */
    
    /* Create source group */
    if ( hwconfig.source_chans > 0)
    {
	src_count = hwconfig.source_chans;
	if (src_count > SRC_MAX)
	    src_count = SRC_MAX;
	if (src_ch != 0)	/* Let user specify number of src ch to use */
	   src_count = src_ch;
	for (i = 0; i < src_count; i++)
	    src_list[i] = E1432_SOURCE_CHAN(i+1);
	src_group = e1432_create_channel_group(hw, src_count, src_list);
	if (src_group >= 0)
	{
	    DEBUG((void) printf("e1432_create_channel_group returned %d\n",
			    src_group));
	    return -1;
	}
    }
    else
    {
	printf("ERROR: No source channels in module at VXI address %d\n",laddr);
	return -1;
    }

    if(verbose) printf("src_count = %d\n", src_count);

    /* Set up interrupts */
    CHECK(irq_setup(laddr, &line));

/* ######## setup sources and start test beginning with highest clock freq */

while(rt_pass < rt_clock_freqs_num)
{
    if(rt_check)
    {
	clock_freq = rt_clock_freqs[rt_pass];
	span = rt_clock_freqs[rt_pass]/(2.56*rt_dec_factor);
    }

    /* initialize source */
    CHECK(e1432_set_clock_freq(hw, src_group, clock_freq));
    if(span==0)
	span = clock_freq/2.56;
    CHECK(e1432_set_span(hw, src_group, span));
    CHECK(e1432_get_span(hw, src_group, &span));
    if(verbose)
	printf("span = %f\n", span);
    CHECK(e1432_set_active(hw, src_group, E1432_CHANNEL_ON));
    CHECK(e1432_set_source_mode(hw, src_group,  source_mode));
    CHECK(e1432_set_range(hw, src_group, 1.0));
    CHECK(e1432_set_amp_scale(hw, src_group, .9995));
    CHECK(e1432_set_amp_scale(hw, src_list[0], .25));
    CHECK(e1432_set_sine_freq(hw, src_group, 1000));

    CHECK(e1432_set_auto_arm(hw, src_group, E1432_AUTO_ARM));
    CHECK(e1432_set_auto_trigger(hw, src_group, E1432_AUTO_TRIGGER));
    src_dutycycle = .5;
    CHECK(e1432_set_duty_cycle(hw, src_group, src_dutycycle));


/* determine blocksize and ramptime */

    /* defaults for not block mode */
    src_ramptime = testtime_max/2;
    src_blocksize = 1024;

    if((source_mode == E1432_SOURCE_MODE_BARB) ||
       (source_mode == E1432_SOURCE_MODE_BSINE) ||
       (source_mode == E1432_SOURCE_MODE_BRAND) ||
       (source_mode == E1432_SOURCE_MODE_BRANDZ))
    {
	src_blocksize = (testtime_max / 10)  * clock_freq/rt_dec_factor;
	if (src_blocksize<16) src_blocksize = 16;
	if (src_blocksize>1024) src_blocksize = 1024;
	src_ramptime = src_dutycycle*(src_blocksize * rt_dec_factor/clock_freq)/2;
    }
    if (src_ramptime>3) src_ramptime = 3;

/* set blocksize, ignored if not in burst mode */

    if(verbose) printf("src_blocksize = %d\n", src_blocksize);
    CHECK(e1432_set_source_blocksize(hw, src_group, src_blocksize));

/* set ramp rate */
    if(rt_ramp)
    {
	if(verbose) printf("src_ramptime = %f\n", src_ramptime);
	CHECK(e1432_set_ramp_rate(hw, src_group,  src_ramptime));
    }
    else
    {
	if(verbose) printf("src_ramptime = 0\n");
	CHECK(e1432_set_ramp_rate(hw, src_group,  0));
    }
    
    
    if((source_mode == E1432_SOURCE_MODE_ARB) || 
	   (source_mode == E1432_SOURCE_MODE_BARB))
    {
           arb_mode = source_buffer_mode;
           for(i=0; i < 4; i++)
           {
	      arb_siglength = arb_srcbufsize;
	      arb_sigperiod = 256;
	      sigtype = 1;
           }
	
           if (arb_xfrsize > arb_srcbufsize) arb_xfrsize = arb_srcbufsize;
	   if (arb_xfrsize > ARBXFRSIZEMAX) arb_xfrsize = ARBXFRSIZEMAX;
	   
           if(source_buffer_mode != E1432_SRCBUFFER_PERIODIC_A)
           {
              arb_xfrmode = E1432_SRC_DATA_MODE_AB;
              arb_xfrmodewait = E1432_SRC_DATA_MODE_WAITAB;
           }
	   else
           {
              arb_xfrmode = E1432_SRC_DATA_MODE_A;
              arb_xfrmodewait = E1432_SRC_DATA_MODE_WAITA;
	   }

           if(verbose) printf("arb_xfrsize %d, arb_srcbufsize %d, \
		arb_siglength %d\n", arb_xfrsize, arb_srcbufsize, 
		arb_siglength);
           if(verbose) printf("arb_xfrmode 0x%x, arb_xfrmodewait %d\n",
                      arb_xfrmode, arb_xfrmodewait);

           for(i=0; i < src_count; i++) 
           {
              CHECK(e1432_set_srcbuffer_mode(hw, src_list[i], arb_mode));
              CHECK(e1432_set_srcbuffer_size(hw, src_list[i], arb_srcbufsize));
              CHECK(e1432_set_srcbuffer_init(hw, src_list[i],
                                       E1432_SRCBUFFER_INIT_EMPTY));
	      CHECK(e1432_get_srcbuffer_size(hw, src_list[i], &temp_long));
	      if(temp_long != arb_srcbufsize)
	      {
		  printf("WARNING!  Source buffer size changed from %d to %d\n",
				arb_srcbufsize, temp_long);
	      }

	   }
           for(i=0; i < src_count; i++) 
           {
	      /* create signal array */
              if(verbose) printf("Creating signal, Ch %d\n", i);
	      src_sig(arb_siglength,2*arb_sigperiod/(2*(i+1)), 
			src_sigbuffptrs_a[i]);
	      src_sig(arb_siglength,2*arb_sigperiod, src_sigbuffptrs_b[i]);
	  }
	   
	   /* down load arb source signal */
	   arb_xfrsize2 = arb_xfrsize;
	   for (k=0; k < arb_srcbufsize; k = k + arb_xfrsize)
	   {
	       arb_srcbufleft = arb_srcbufsize - k;
	       if (arb_srcbufleft < arb_xfrsize) arb_xfrsize2 = arb_srcbufleft;
	       for(i=0; i < src_count; i++) 
	       {
		   if (e1432_check_src_arbrdy(hw, src_list[i], arb_xfrmodewait))
		   {
		       if(verbose) printf("Start pre-loading buffer A, Ch %d\n", i);
		       xfr_src_sig(hw, src_list[i], src_sigbuffptrs_a[i], 
				   arb_xfrsize, arb_xfrmodewait);
		   }
		   if(verbose) printf("done pre-loading buffer A, Ch %d\n", i);
	       }
	   }
	   
	   arb_xfrsize2 = arb_xfrsize;
	   for (k=0; k < arb_srcbufsize; k = k + arb_xfrsize)
	   {
	       arb_srcbufleft = arb_srcbufsize - k;
	       if (arb_srcbufleft < arb_xfrsize) arb_xfrsize2 = arb_srcbufleft;
	       for(i=0; i < src_count; i++) 
	       {
		   if (e1432_check_src_arbrdy(hw, src_list[i], arb_xfrmodewait))
		   {
		       if(verbose) printf("pre-loading buffer B\n");
		       /* load B as well as A */
		       xfr_src_sig(hw, src_list[i], src_sigbuffptrs_b[i], 
				   arb_xfrsize, arb_xfrmodewait);
		   }
		   if(verbose) printf("done pre-loading buffer A, Ch %d\n", i);
	       }

	   }
       }
    
    CHECK(e1432_set_interrupt_priority(hw, src_group, line));
    CHECK(e1432_set_interrupt_mask(hw, src_group, E1432_IRQ_SRC_STATUS));

    CHECK(gettimeofday(&tv1, &tz));

    /* Start measurement */
    if(verbose) printf("init_measure\n");
   
    error = e1432_init_measure(hw, src_group);
    if(error == ERR1432_SRC_REALTIME_RESTRICTION)
    {
       printf("Error: e1432_init_measure returned %d\n", error); 
       underrun = 1;
    }
    else if (error != 0)
    {
       printf("e1432_init_measure returned error %d\n", error);
       return error;
    }

    /* Sleep for a long time.  Note that receiving an interrupt aborts
       the sleep, so we must keep calling it. */
    for ( ; ; )
    {
        if(!irq_flag)
           if(!rt_check && source_mode != E1432_SOURCE_MODE_ARB)
           {
              if(key_pressed())
              {
                 CHECK(e1432_get_amp_scale(hw, src_list[0], &amp));
                 if(verbose)
	            printf("Ch 1 amp = %f, Setting Ch 1 amp = %f\n", amp, 
			(float)1.0 - amp);
                 CHECK(e1432_set_amp_scale(hw, src_list[0], ((float)1.0)-amp));
              }
           }
	   if(rt_otf)	/* Change parms on the fly */
           {
              CHECK(e1432_get_amp_scale(hw, src_list[0], &amp));
              CHECK_RT(e1432_set_amp_scale(hw, src_list[0], ((float)1.0)-amp));
              CHECK(e1432_get_amp_scale(hw, src_list[0], &amp));
              CHECK_RT(e1432_set_amp_scale(hw, src_list[0], ((float)1.0)-amp));
              CHECK(e1432_get_amp_scale(hw, src_list[0], &amp));
              CHECK_RT(e1432_set_amp_scale(hw, src_list[0], ((float)1.0)-amp));
              CHECK(e1432_get_amp_scale(hw, src_list[0], &amp));
              CHECK_RT(e1432_set_amp_scale(hw, src_list[0], ((float)1.0)-amp));
	   }
           
	irq_flag = 0;

        /* Check for timeout outside of interrupt handler if not ARB mode
           since we may never get an interrupt. */
        if((source_mode != E1432_SOURCE_MODE_ARB) || 
	   (source_buffer_mode != E1432_SRCBUFFER_CONTINUOUS) ) 
        {
           (void) gettimeofday(&tv2, & tz);
           time1 = tv2.tv_sec-tv1.tv_sec + (tv2.tv_usec-tv1.tv_usec)/1000000.0;
        }

	/* Exit loop if underrun has occurred or testtime_max period exceeded */
	if((underrun || (time1 > testtime_max)) && rt_check)
	   break;
    }

    CHECK(e1432_reset_measure(hw, src_group));

    if(!underrun && (time1 > testtime_max))
	break;
    rt_pass++;
    underrun = 0;	/* Reset RT indicator */
}

printf("Ch:%d,Mode:%s,Dec:%d,OTF(scale):%d,Ramp:%d,Buf:%d,Xfr:%d,", src_count, mode_ascii[rt_mode-1], rt_dec_factor, rt_otf, rt_ramp,arb_srcbufsize,arb_xfrsize);
if(rt_pass >= rt_clock_freqs_num)
{
   printf("\nERROR: RT not possible!!: Min clock freq tried was %f Hz\n", 
       	rt_clock_freqs[rt_pass - 1]);
}
else
{
    printf("Max clk:%6.0f\n", rt_clock_freqs[rt_pass]);
}

    return 0;
}

